define([
    'underscore',
    'modules/form/section-view',
    'modules/new-appointment-request/views/direct/pac-team/section-view',
    'json!modules/new-appointment-request/views/common/facility/data/facility-section.json',
], function(_, SectionView, PactSectionView, metadata) {
    'use strict';


    var EXPRESS_CARE_ID = 'CR1';
    var PRIMARY_CARE_ID = '323';


    /**
     * @class FacilitySectionView
     * @typedef {Backbone.View<Backbone.Model>}
     */
    return SectionView.extend({
        collectionEvents: {'change:value': 'updateFDNS   el'},

        ui: {express: '.express-time'},

        initialize: function(options) {
            var sectionName = 'facility-section';
            this.legend = this.getLegendFromMetadata(sectionName, metadata, false) || '';
            this.collection = this.getQuestionsFromMetadata(sectionName, metadata);
            this.facilities = options.patientEnrolledFacilities;
            this.clinicsInstitutionCodes = options.clinicsInstitutionCodes || [];
            this.googleAnalytics = options.googleAnalyticsProvider.getInstance();

            this.setFacilityQuestionModel();
            this.initializeQuestionEventListeners();

            this.listenTo(this.model.pacTeam(), 'fetch:success', this.injectPacTeam);
        },

        onRender: function() {
            var $field = this.$('.field-contain');
            $field.append($('<p class="express-time">'));
            this.bindUIElements();
        },

        onBeforeDestroy: function() {
            // This should prob go in the section view
            this.collection.off('change:value');
        },

        setFacilityQuestionModel: function() {
            var facilities = this.facilities.toSectionData(this.clinicsInstitutionCodes);
            var facilityQuestionModel = this.collection.get('facility');
            facilities = _.chain(facilities)
                .sortBy('text')
                .sortBy(function(facility) {
                    return facility.hasPactLabel * -1;
                })
                .value();
            facilityQuestionModel.set('values', facilities);
        },

        updateFDNS   el: function(model) {
            var id = model.get('id');
            var value = model.get('value');
            var facilityModel;
            var facilityName;

            // Remove view for PAC team for conditional injecting
            if (this.model.getCareId() === PRIMARY_CARE_ID) {
                $('.pacTeam').remove();
            }

            if (!_.isUndefined(value)) {
                if (typeof value === 'string') {
                    facilityModel = this.facilities.findWhere({institutionCode: value});
                } else {
                    facilityModel = this.facilities.findWhere({institutionCode: value.get('institutionCode')});
                }

                this.model.set(id, facilityModel);
                this._insertExpressCareMessage(facilityModel);
                facilityName = facilityModel.formatTextValue(false);

                // This was originally in type of care in the old Google Analytics, but because they where
                // sending the facility name, It has been moved to here
                this.googleAnalytics.gas('send', 'event', 'veteran-appointment', 'facility-selected', facilityName);
            }
        },

        /**
         * Appends the DOM with an expressCare message
         * @param {Backbone.Model} facilityModel
         * @return {void}
         * @private
         */
        _insertExpressCareMessage: function(facilityModel) {
            var isExpressCare = this.model.getCareId() === EXPRESS_CARE_ID;
            var message;

            if (this.ui.express) {
                this.ui.express.empty();
            }

            if (isExpressCare && facilityModel.isExpressSupportedToday()) {
                message = facilityModel.getExpressMessage();
                this.ui.express.text(message);
            }
        },

        injectPacTeam: function() {
            var facility = this.model.facility();
            var pacTeam = this.model.pacTeam();
            var siteCode = this.model.getSiteCode();

            if (this.pacTeamView) {
                this.pacTeamView.destroy();
                this.pacTeamView = null;
            }

            if (pacTeam.hasPacTeamAtFacility(siteCode)) {
                this.$el.css('display', 'flex');
                this.pacTeamView = new PactSectionView({
                    model: this.model,
                    team: pacTeam,
                    facility: facility,
                });
                this.pacTeamView.render();
                this.pacTeamView.$el.children().each(function () {
                    $(this).css({'margin-top': 0, 'margin-bottom': 0, 'padding': 0});
                });
                this.pacTeamView.$('p').css({'margin-top': 0, 'padding': 0});
                this.$el.append(this.pacTeamView.$el);
            }
        },
    });
});
